#!/bin/bash
set -euo pipefail

# =========================
# Zeit & Host
# =========================
TS=$(date +"%Y-%m-%d_%H-%M")
HOST=$(hostname -s)

# =========================
# Verzeichnisse
# =========================
BACKUP_BASE="<BACKUP_BASE_DIR>"          # z. B. /var/backups/ha
TMP="/tmp/ha-backup-$TS"
ARCHIVE="$BACKUP_BASE/ha-backup-$HOST-$TS.tar.gz"

# =========================
# Zu sichernde Pfade
# =========================
HA_CFG="/home/<USER>/.homeassistant"
PIHOLE_ETC="/etc/pihole"
DNSMASQ_ETC="/etc/dnsmasq.d"

# =========================
# Nextcloud / WebDAV (optional)
# =========================
WEBDAV_URL="https://<NEXTCLOUD_HOST>/remote.php/dav/files/<USER>/Backups/ha/"
WEBDAV_ROOT="https://<NEXTCLOUD_HOST>/remote.php/dav/files/<USER>"
NETRC="/root/.netrc"

# =========================
# Vorbereitung
# =========================
mkdir -p "$TMP" "$BACKUP_BASE"

# =========================
# Home Assistant
# =========================
echo "→ Backup Home Assistant config"
rsync -a --delete \
  --exclude "home-assistant_v2.db*" \
  "$HA_CFG/" "$TMP/homeassistant/"

# =========================
# Zigbee
# =========================
echo "→ Backup Zigbee (Zigbee2MQTT/ZHA)"

# 1) ZHA: liegt in HA Config (.storage) -> wird durch HA rsync mitgesichert
if [[ -d "$HA_CFG/.storage" ]]; then
  echo "  - ZHA (.storage) ist Teil der HA-Sicherung"
fi

# 2) Zigbee2MQTT: typische Datenpfade (abhängig vom Setup)
mkdir -p "$TMP/zigbee2mqtt"

for Z2M in \
  "$HA_CFG/zigbee2mqtt" \
  "$HA_CFG/zigbee2mqtt/data" \
  "/opt/zigbee2mqtt/data" \
  "/var/lib/zigbee2mqtt" \
  "/srv/zigbee2mqtt/data" \
; do
  if [[ -d "$Z2M" ]]; then
    echo "  - Sicherung Zigbee2MQTT Daten aus: $Z2M"
    rsync -a "$Z2M/" \
      "$TMP/zigbee2mqtt/$(echo "$Z2M" | tr '/' '_' | sed 's/^_//')/"
  fi
done

# Koordinator-Backup-Dateien (sehr wichtig für Restore)
find "$TMP/zigbee2mqtt" -maxdepth 4 -type f \
  \( -name "coordinator_backup.json" -o -name "backup_*.json" \) \
  -print 2>/dev/null || true

# =========================
# Pi-hole
# =========================
echo "→ Backup Pi-hole"
mkdir -p "$TMP/pihole/etc-pihole" "$TMP/pihole/etc-dnsmasq.d"

rsync -a "$PIHOLE_ETC/" "$TMP/pihole/etc-pihole/" || true
rsync -a "$DNSMASQ_ETC/" "$TMP/pihole/etc-dnsmasq.d/" || true

# =========================
# Archiv
# =========================
echo "→ Archive erstellen"
tar -czf "$ARCHIVE" -C "$TMP" .

rm -rf "$TMP"

# =========================
# Upload (optional)
# =========================
echo "→ Optional: Upload zu Nextcloud"
if [[ -f "$NETRC" ]]; then
  curl -fsS --netrc-file "$NETRC" -T "$ARCHIVE" "$WEBDAV_URL"
fi

echo "✔ Backup fertig: $ARCHIVE"

# =========================
# Retention lokal
# =========================
echo "→ Retention: lokale Backups begrenzen (<LOCAL_RETENTION>)"

ls -1t "$BACKUP_BASE"/ha-backup-*.tar.gz 2>/dev/null \
  | tail -n +<LOCAL_RETENTION_PLUS_ONE> \
  | xargs -r rm -f

# =========================
# Retention Nextcloud
# =========================
echo "→ Retention: Nextcloud Backups begrenzen (<REMOTE_RETENTION>)"

curl -fsS --netrc-file "$NETRC" -X PROPFIND -H "Depth: 1" \
  "${WEBDAV_ROOT}/Backups/ha/" \
| grep -oE 'ha-backup-[^<]+' \
| sort -r \
| tail -n +<REMOTE_RETENTION_PLUS_ONE> \
| while read -r OLD; do
    curl -fsS --netrc-file "$NETRC" -X DELETE \
      "${WEBDAV_ROOT}/Backups/ha/${OLD}" || true
  done

